/*

 @ author, GLSL & OpenGL code author Athanasios Theocharidis, 2009-2015

*/

vec2 randSeed = vec2(0.0);
void initRand(in vec2 seed);
float rand1();
float rand1(in vec2 seed);
vec2 rand2();
vec2 rand2(in vec2 seed);
vec3 rand3();
vec3 rand3(in vec2 seed);
vec3 cosHemiRandom();
vec3 cosHemiRandom(in vec2 seed);
vec3 cosPowHemiRandom(float k);
vec3 cosPowHemiRandom(in vec2 seed, float k);

void initRand(in vec2 seed)
{
  randSeed = seed;
}

// This function returns uniformly distributed values in the range [0, 1]
float rand1()
{
  float dotProduct = dot(randSeed, vec2(12.9898,78.233));
  float x = fract(sin(dotProduct) * 43758.5453);
  randSeed.x = x;
  return x;
}

// This function returns uniformly distributed values in the range [0, 1]
float rand1(in vec2 seed)
{
  float dotProduct = dot(seed, vec2(12.9898,78.233));
  float x = fract(sin(dotProduct) * 43758.5453);
  return x;
}

// This function returns uniformly distributed values in the range [0, 1]
vec2 rand2()
{
  float dotProduct = dot(randSeed, vec2(12.9898,78.233));
  vec2 r = vec2(fract(sin(dotProduct ) * 43758.5453), fract(sin(2.0 * dotProduct) * 43758.5453));
  randSeed = r;
  return r;
}

// This function returns uniformly distributed values in the range [0, 1]
vec2 rand2(in vec2 seed)
{
  float dotProduct = dot(seed, vec2(12.9898,78.233));
  vec2 r = vec2(fract(sin(dotProduct ) * 43758.5453), fract(sin(2.0 * dotProduct) * 43758.5453));
  return r;
}

// This function returns uniformly distributed values in the range [0, 1]
vec3 rand3()
{
  float dotProduct = dot(randSeed, vec2(12.9898,78.233));
  vec3 r = vec3(fract(sin(dotProduct ) * 43758.5453), fract(sin(2.0 * dotProduct) * 43758.5453), fract(sin(3.0 * dotProduct) * 43758.5453));
  randSeed = r.xy;
  return r;
}

// This function returns uniformly distributed values in the range [0, 1]
vec3 rand3(in vec2 seed)
{
  float dotProduct = dot(seed, vec2(12.9898,78.233));
  vec3 r = vec3(fract(sin(dotProduct ) * 43758.5453), fract(sin(2.0 * dotProduct) * 43758.5453), fract(sin(3.0 * dotProduct) * 43758.5453));
  return r;
}

// Returns 3D unit vectors distributed according to a
// cosine distribution around the Z axis.
vec3 cosHemiRandom()
{
  vec2 e = rand2();

  // Jensen's method
  float sinTheta = sqrt(1.0 - e.x);
  float cosTheta = sqrt(e.x);
  float phi = 6.28318531 * e.y;

  // We could also use Malley's method (pbrt p.657), since they have the same cost:
  // r = sqrt(e.x);
  // t = 2*pi*e.y;
  // x = cos(t)*r;
  // y = sin(t)*r;
  // z = sqrt(1.0 - x*x + y*y);

  return vec3(cos(phi) * sinTheta, sin(phi) * sinTheta, cosTheta);
}

// Returns 3D unit vectors distributed according to a
// cosine distribution around the Z axis.
vec3 cosHemiRandom(in vec2 seed)
{
  vec2 e = rand2(seed);

  // Jensen's method
  float sinTheta = sqrt(1.0 - e.x);
  float cosTheta = sqrt(e.x);
  float phi = 6.28318531 * e.y;

  // We could also use Malley's method (pbrt p.657), since they have the same cost:
  // r = sqrt(e.x);
  // t = 2*pi*e.y;
  // x = cos(t)*r;
  // y = sin(t)*r;
  // z = sqrt(1.0 - x*x + y*y);

  return vec3(cos(phi) * sinTheta, sin(phi) * sinTheta, cosTheta);
}

// Returns 3D unit vectors distributed according to a cosine
// power distribution (pow(cos(theta), k) around the Z axis.
// The method comes from the G3D engine (http://g3d.sourceforge.net)
// The returned vector is closer to vec3(0.0, 0.0, 1.0) as k becomes big.
vec3 cosPowHemiRandom(float k)
{
  vec2 e = rand2();

  float cosTheta = pow(e.x, 1.0 / (k + 1.0));
  float sinTheta = sqrt(1.0 - cosTheta * cosTheta);
  float phi = 6.28318531 * e.y;

  return vec3(cos(phi) * sinTheta, sin(phi) * sinTheta, cosTheta);
}

// Returns 3D unit vectors distributed according to a cosine
// power distribution (pow(cos(theta), k) around the Z axis.
// The method comes from the G3D engine (http://g3d.sourceforge.net)
// The returned vector is closer to vec3(0.0, 0.0, 1.0) as k becomes big.
vec3 cosPowHemiRandom(in vec2 seed, float k)
{
  vec2 e = rand2(seed);

  float cosTheta = pow(e.x, 1.0 / (k + 1.0));
  float sinTheta = sqrt(1.0 - cosTheta * cosTheta);
  float phi = 6.28318531 * e.y;

  return vec3(cos(phi) * sinTheta, sin(phi) * sinTheta, cosTheta);
}