/*

 @ author, OpenCL code author Athanasios Theocharidis, 2010-2011

*/

#ifdef __USE_PLASMA_KERNEL__
    inline char4 plasmaVolume(const uint, const uint, const uint, __global const int *, __global const int *, __global const int *, __global const char *);
#elif defined(__USE_CLOUD_KERNEL__)
    inline char4 cloudVolume(const uint, const uint, const uint, __global const int *, __global const int *, __global const int *, __global const char*, const int, const int, const int);
    inline int evaluateCloudSines(const uint, const uint, const uint, int, __global const int *, __global const int *, __global const int *);
#elif defined(__USE_VORONOI_KERNEL__)
    #if USE_LOCAL_CACHE
        inline char4 voronoiVolume(const uint, const uint, const uint, __local const int*, __local const int*, __local const int*, __global const char*, const int);
    #else
        inline char4 voronoiVolume(const uint, const uint, const uint, __global const int*, __global const int*, __global const int*, __global const char*, const int);
    #endif
#endif
inline char4 getPalletteColorAndSetToVolume(int, __global const char*);
inline char4 fadeByDistanceFromVolumeCenter(char4, const uint, const uint, const uint, const int, const int, const int, const float);

__kernel void volume3DTexture(__global char *volume3DTextureArray, __global const int *arrayX, __global const int *arrayY, __global const int *arrayZ, __global const char *rgbaColorPaletteArray,
                              const int width, const int height, const int depth, const int useFadeByDistance, const float fadeByDistanceRadius
#if USE_LOCAL_CACHE
                            , const int voronoiIterationSizeLocalIndex, __local int *localArrayX, __local int *localArrayY, __local int *localArrayZ)
#else
                              )
#endif
{
    const uint x = get_global_id(0);
    const uint y = get_global_id(1);
    const uint z = get_global_id(2);
    const uint localX = get_local_id(0);

    #if USE_LOCAL_CACHE
        if (localX < voronoiIterationSizeLocalIndex)
        {
            #ifdef __USE_VECTOR8_COPY__
                vstore8(vload8(localX, arrayX), localX, localArrayX);
                vstore8(vload8(localX, arrayY), localX, localArrayY);
                vstore8(vload8(localX, arrayZ), localX, localArrayZ);
           #elif defined(__USE_VECTOR16_COPY__)
                vstore16(vload16(localX, arrayX), localX, localArrayX);
                vstore16(vload16(localX, arrayY), localX, localArrayY);
                vstore16(vload16(localX, arrayZ), localX, localArrayZ);
           #endif
        }
        barrier(CLK_LOCAL_MEM_FENCE);
    #endif

    if ( (x < width) && (y < height) && (z < depth) )
    {
        int index = (z * width * height + y * width + x);
        #ifdef __USE_PLASMA_KERNEL__
            char4 rgba = plasmaVolume(x , y, z, arrayX, arrayY, arrayZ, rgbaColorPaletteArray);
        #elif defined(__USE_CLOUD_KERNEL__)
            char4 rgba = cloudVolume(x , y, z, arrayX, arrayY, arrayZ, rgbaColorPaletteArray, width, height, depth);
        #elif defined(__USE_VORONOI_KERNEL__)
            #if USE_LOCAL_CACHE
                char4 rgba = voronoiVolume(x , y, z, localArrayX, localArrayY, localArrayZ, rgbaColorPaletteArray, width);
            #else
                char4 rgba = voronoiVolume(x , y, z, arrayX, arrayY, arrayZ, rgbaColorPaletteArray, width);
            #endif
        #endif

        if (useFadeByDistance)
            rgba = fadeByDistanceFromVolumeCenter(rgba, x, y, z, width, height, depth, fadeByDistanceRadius);

        vstore4(rgba, index, volume3DTextureArray);
    }
}

#ifdef __USE_PLASMA_KERNEL__
    inline char4 plasmaVolume(const uint x, const uint y, const uint z, __global const int *fsin1, __global const int *fsin2, __global const int *fsin3, __global const char *rgbaColorPaletteArray)
    {
        int timer = 0;
        int c = (fsin1[x] + fsin2[y] + fsin3[z] + fsin1[x + y + timer] + fsin2[x + z + (timer << 1)] + fsin3[y + z + (timer << 2)]) & 255;

        return getPalletteColorAndSetToVolume(c, rgbaColorPaletteArray);
    }
#elif defined(__USE_CLOUD_KERNEL__)
    inline char4 cloudVolume(const uint x, const uint y, const uint z, __global const int *fsin1, __global const int *fsin2, __global const int *fsin3, __global const char *rgbaColorPaletteArray, const int width, const int height, const int depth)
    {
        int evalSum = 0;
        int octave = CLOUD_OCTAVE_SIZE;
        while (--octave >= 0)
            evalSum += evaluateCloudSines(x, y, z, octave + 1, fsin1, fsin2, fsin3);

        int c = evalSum & 511;
        if ( (x == 0) || (x == width - 1) || (y == 0) || (y == height - 1) || (z == 0) || (z == depth - 1) ) c = 0;
        if (c > 255) c = 511 - c;

        return getPalletteColorAndSetToVolume(c, rgbaColorPaletteArray);
    }

    inline int evaluateCloudSines(const uint x, const uint y, const uint z, int octave, __global const int *fsin1, __global const int *fsin2, __global const int *fsin3)
    {
        return (fsin1[(x << octave) & CLOUD_FSIN_SIZE_MODULO] >> octave) + (fsin2[(y << octave) & CLOUD_FSIN_SIZE_MODULO] >> octave) + (fsin3[(z << octave) & CLOUD_FSIN_SIZE_MODULO] >> octave) + (fsin3[((x + y + 64) << octave) & CLOUD_FSIN_SIZE_MODULO] >> octave);
    }
#elif defined(__USE_VORONOI_KERNEL__)
    #if USE_LOCAL_CACHE
        inline char4 voronoiVolume(const uint x , const uint y, const uint z, __local const int *arrayX, __local const int *arrayY, __local const int *arrayZ, __global const char *rgbaColorPaletteArray, const int width)
    #else
        inline char4 voronoiVolume(const uint x , const uint y, const uint z, __global const int *arrayX, __global const int *arrayY, __global const int *arrayZ, __global const char *rgbaColorPaletteArray, const int width)
    #endif
    {
        float dist = 0.0f;
        float min = 0x1.fffffeP+127f; // 3.4028235e+38f -> Float.MAX_VALUE
        int iteration = VORONOI_ITERATION_SIZE;
        while (--iteration >= 0)
        {
            // dist = fast_length( (float4)(x - arrayX[iteration], y - arrayY[iteration], z - arrayZ[iteration], 0.0f) );
            // dist = sqrt( ( mad( (float)(x - arrayX[iteration]), (float)(x - arrayX[iteration]), mad( (float)(y - arrayY[iteration]), (float)(y - arrayY[iteration]), mad( (float)(z - arrayZ[iteration]), (float)(z - arrayZ[iteration]), 0 ) ) ) ) );
            dist = sqrt( (float)( (x - arrayX[iteration]) * (x - arrayX[iteration]) + (y - arrayY[iteration]) * (y - arrayY[iteration]) + (z - arrayZ[iteration]) * (z - arrayZ[iteration]) ) );
            if (dist < min)
                min = dist;
        }

        float cc = 2.0f * min;
        if (cc < 64.0f) cc /= 2.0f;
        if (cc > width) cc = (float)width;
        int c = (int)( (cc / (float)width) * 255 ) & 255;

        return getPalletteColorAndSetToVolume(c, rgbaColorPaletteArray);
    }
#endif

inline char4 getPalletteColorAndSetToVolume(int c, __global const char *rgbaColorPaletteArray)
{
    return vload4(c, rgbaColorPaletteArray);
}

inline char4 fadeByDistanceFromVolumeCenter(char4 rgba, const uint x, const uint y, const uint z, const int width, const int height, const int depth, const float fadeByDistanceRadius)
{
    float4 distanceCenter = (float4)(x - width / 2.0f, y - height / 2.0f, z - depth / 2.0f, 0.0f);
    float c = fadeByDistanceRadius * ( width / 2.0f - fast_length(distanceCenter) );
    if (c < 0.0f) c = 0.0f;
    if (c > width) c = (float)width;
    float cc = c / (float)width;

    return convert_char4(convert_float4(rgba) * cc);
}